#!/xbin/sh

# Default input parameters
MEASUREMENT_TIME=30
TEL_NUMBER="*054"

SERIAL_ANSWER=""
SERIAL_CLIENT_WAIT_STEP=100000
SERIAL_CLIENT_PATH=/marvell/tel
COMMPM_ACC_PATH=/marvell/tel
CALL_WAIT_TIME=30000000
ATH_WAIT_TIME=5000000
GSM_SCENARIO_FAIL=0
WB_SCENARIO_FAIL=0

# Paths
TEMP_DIR=/mrvlsys
FULL_LOG_FILE=${TEMP_DIR}/pm_sanity_log
TMP_OPS=${TEMP_DIR}/ops
TMP_APPS_DUTYCYCLE=${TEMP_DIR}/tmp_duty_cycle
TMP_TIMERS_STATS=${TEMP_DIR}/tmp_timer_stats
TMP_COMM_DUTYCYCLE=${TEMP_DIR}/tmp_rpc
TMP_FIFOIN=${TEMP_DIR}/fifoin
TMP_SERIAL_ANSWER=${TEMP_DIR}/ser_file
TMP_OP=${TEMP_DIR}/tmp_op
TMP_DVFM_TEST_ANSWER=${TEMP_DIR}/dvfm_answ
FINAL_RESUALTS_FILE=${TEMP_DIR}/sanity_test_result
TMP_INTERM_DVFM_TEST_ANSWER=${TEMP_DIR}/interm_dvfm_answ

#scenarios
#Execute all scenarios by default
DRX_SCENARIO=1
GSM_SCENARIO=1
DRX_AFTER_GSM_SCENARIO=1
WB_SCENARIO=1
DRX_AFTER_WB_SCENARIO=1
DVFM_SCENARIO=1

#total test result
SANITY_RESULT="PASS"

# FAIL Targets
DRX_FAIL_TARGET_APPS_D2_IDLE=10.00
DRX_FAIL_TARGET_APPS_TOTAL_IDLE=10.00
DRX_FAIL_TARGET_APPS_TIMERS=100.00
DRX_FAIL_TARGET_COMM_SG_PP_IDLE=00.00
DRX_FAIL_TARGET_COMM_D2_IDLE=10.00
DRX_FAIL_TARGET_COMM_DSP_PP_IDLE=10.00

GSM_FAIL_TARGET_APPS_CG_IDLE=10.00
GSM_FAIL_TARGET_APPS_TOTAL_IDLE=10.00
GSM_FAIL_TARGET_APPS_TIMERS=100.00
GSM_FAIL_TARGET_COMM_SG_PP_IDLE=10.00
GSM_FAIL_TARGET_COMM_DSP_PP_IDLE=10.00

WB_FAIL_TARGET_APPS_CG_IDLE=10.00
WB_FAIL_TARGET_APPS_TOTAL_IDLE=10.00
WB_FAIL_TARGET_APPS_TIMERS=100.00
WB_FAIL_TARGET_COMM_SG_PP_IDLE=10.00
WB_FAIL_TARGET_COMM_DSP_PP_IDLE=10.00

# Targets
DRX_TARGET_APPS_D2_IDLE=90.00
DRX_TARGET_APPS_TOTAL_IDLE=95.00
DRX_TARGET_APPS_TIMERS=20.00
DRX_TARGET_COMM_SG_PP="PP1"
DRX_TARGET_COMM_SG_PP_IDLE=00.00
DRX_TARGET_COMM_D2_IDLE=92.00
DRX_TARGET_COMM_DSP_PP="PP1"
DRX_TARGET_COMM_DSP_PP_IDLE=92.00

GSM_TARGET_APPS_CG_IDLE=95.00
GSM_TARGET_APPS_TOTAL_IDLE=95.00
GSM_TARGET_APPS_TIMERS=20.00
GSM_TARGET_COMM_SG_PP="PP1"
GSM_TARGET_COMM_SG_PP_IDLE=70.00
GSM_TARGET_COMM_DSP_PP="PP1"
GSM_TARGET_COMM_DSP_PP_IDLE=40.00

WB_TARGET_APPS_CG_IDLE=95.00
WB_TARGET_APPS_TOTAL_IDLE=95.00
WB_TARGET_APPS_TIMERS=20.00
WB_TARGET_COMM_SG_PP="PP2"
WB_TARGET_COMM_SG_PP_IDLE=70.00
WB_TARGET_COMM_DSP_PP="PP2"
WB_TARGET_COMM_DSP_PP_IDLE=40.00

# Names for result file
DRX_NAME_APPS_D2_IDLE="DRX APPS D2 IDLE"
DRX_NAME_APPS_TOTAL_IDLE="DRX APPS TOTAL IDLE"
DRX_NAME_COMM_SG_D2_IDLE="DRX COMM SG D2 IDLE"
DRX_NAME_APPS_TIMERS="DRX APPS TIMERS"
DRX_NAME_COMM_SG_PP="DRX COMM SG PP"
DRX_NAME_COMM_SG_PP_IDLE="DRX COMM SG PP IDLE"
DRX_NAME_COMM_DSP_PP="DRX COMM DSP PP"
DRX_NAME_COMM_DSP_PP_IDLE="DRX COMM DSP PP IDLE"

GSM_NAME_APPS_CG_IDLE="GSM VC APPS CG IDLE"
GSM_NAME_APPS_TOTAL_IDLE="GSM VC APPS TOTAL IDLE"
GSM_NAME_COMM_SG_PP="GSM VC COMM SG PP"
GSM_NAME_COMM_SG_PP_IDLE="GSM VC COMM SG PP IDLE"
GSM_NAME_COMM_DSP_PP="GSM VC COMM DSP PP"
GSM_NAME_COMM_DSP_PP_IDLE="GSM VC COMM DSP PP IDLE"
GSM_NAME_APPS_TIMERS="GSM APPS TIMERS"

DRX_AFTER_GSM_NAME="DRX AFTER GSM"

WB_NAME_APPS_CG_IDLE="WB VC APPS CG IDLE"
WB_NAME_APPS_TOTAL_IDLE="WB VC APPS TOTAL IDLE"
WB_NAME_COMM_SG_PP="WB VC COMM SG PP"
WB_NAME_COMM_SG_PP_IDLE="WB VC COMM SG PP IDLE"
WB_NAME_COMM_DSP_PP="WB VC COMM DSP PP"
WB_NAME_COMM_DSP_PP_IDLE="WB VC COMM DSP PP IDLE"
WB_NAME_APPS_TIMERS="WB APPS TIMERS"

DRX_AFTER_WB_NAME="DRX AFTER WB"

##################################################################################
#
# Input:		$1 - name of test
#
# Return value:	Value of the test
#
# Description:	Parse targets file, find selected field by test name and return
#				it's numeric results
#
##################################################################################
parse_value () {
busybox awk -v test_name="$1" '
	{
		if(match($0, test_name))
		{
			s=$(NF-1);
			sub("%", "",s);
			print s;
		}
	}' $TARGETS_FILE
}

##################################################################################
#
# Input:		$1 - name of test
#
# Return value:	PP value of the test
#
# Description:	Parse targets file, find selected field by test name and return
#				it's pp results
#
##################################################################################
parse_pp () {
busybox awk -v test_name="$1" '
	{
		if(match($0, test_name))
		{
			s=$(NF-1);
			if(match(s, "PP"))
				print s;
		}
	}' $TARGETS_FILE
}

##################################################################################
#
# Input:		None
#
# Return value:	None
#
# Description:	Set targets according to last results file
#
##################################################################################
set_targets () {
	DRX_TARGET_APPS_D2_IDLE=`parse_value "$DRX_NAME_APPS_D2_IDLE"`
	DRX_TARGET_APPS_TOTAL_IDLE=`parse_value "$DRX_NAME_APPS_TOTAL_IDLE"`
	DRX_TARGET_APPS_TIMERS=`parse_value "$DRX_NAME_APPS_TIMERS"`
	DRX_TARGET_COMM_D2_IDLE=`parse_value "$DRX_NAME_COMM_SG_D2_IDLE"`
	DRX_TARGET_COMM_SG_PP=`parse_pp "$DRX_NAME_COMM_SG_PP"`
	DRX_TARGET_COMM_SG_PP_IDLE=`parse_value "$DRX_NAME_COMM_SG_PP_IDLE"`
	DRX_TARGET_COMM_DSP_PP=`parse_pp "$DRX_NAME_COMM_DSP_PP`
	DRX_TARGET_COMM_DSP_PP_IDLE=`parse_value "$DRX_NAME_COMM_DSP_PP_IDLE"`

	GSM_TARGET_APPS_CG_IDLE=`parse_value "$GSM_NAME_APPS_CG_IDLE"`
	GSM_TARGET_APPS_TOTAL_IDLE=`parse_value "$GSM_NAME_APPS_TOTAL_IDLE"`
	GSM_TARGET_COMM_SG_PP=`parse_pp "$GSM_NAME_COMM_SG_PP"`
	GSM_TARGET_COMM_SG_PP_IDLE=`parse_value "$GSM_NAME_COMM_SG_PP_IDLE"`
	GSM_TARGET_COMM_DSP_PP=`parse_pp "$GSM_NAME_COMM_DSP_PP"`
	GSM_TARGET_COMM_DSP_PP_IDLE=`parse_value "$GSM_NAME_COMM_DSP_PP_IDLE"`
	GSM_TARGET_APPS_TIMERS=`parse_value "$GSM_NAME_APPS_TIMERS"`

	WB_TARGET_APPS_CG_IDLE=`parse_value "$WB_NAME_APPS_CG_IDLE"`
	WB_TARGET_APPS_TOTAL_IDLE=`parse_value "$WB_NAME_APPS_TOTAL_IDLE"`
	WB_TARGET_COMM_SG_PP=`parse_pp "$WB_NAME_COMM_SG_PP"`
	WB_TARGET_COMM_SG_PP_IDLE=`parse_value "$WB_NAME_COMM_SG_PP_IDLE"`
	WB_TARGET_COMM_DSP_PP=`parse_pp "$WB_NAME_COMM_DSP_PP"`
	WB_TARGET_COMM_DSP_PP_IDLE=`parse_value "$WB_NAME_COMM_DSP_PP_IDLE"`
	WB_TARGET_APPS_TIMERS=`parse_value "$WB_NAME_APPS_TIMERS"`
}

##################################################################################
#
# Input:		1. Command string
#				2. Acknowledge string
#				3. Waiting time in microsecond
#				4. Fail string1
#				5. Fail string2
#
# Return value:	Status of serial_client command execution
#
# Description:	Start serial_client, push a commands to serial_client and wait for
#				serial_client answer
#				Analyze serial_client answer and returns "OK" or "ERROR" as result
#
##################################################################################
serial_command() {
	busybox usleep $SERIAL_CLIENT_WAIT_STEP
	$SERIAL_CLIENT_PATH/serial_client < $TMP_FIFOIN >$TMP_SERIAL_ANSWER&
	busybox printf "%s\r" $1>$TMP_FIFOIN

	# parse answer until timeout or one of the strings comes: ACK or FAIL1 or FAIL2
	WAITING_TIME=0
	while [ 1 ]
	do
		SERIAL_ANSWER=`busybox awk -v fail_str1="$4" -v ack_str="$2" -v fail_str2="$5" '
		BEGIN
		{
			ACKNOWLEDGE=0
			FAILS=0
		}
		{
			if(match($0, ack_str))
			{
				ACKNOWLEDGE=1
			}
			if(match($0, fail_str1))
			{
				FAILS=1
			}
			if (length(fail_str2)>0)
			{
				if(match($0, fail_str2))
				{
					FAILS=1
				}
			}
		}
		END
		{
			if(FAILS==1)
			{
				print "ERROR"
			}
			else if(ACKNOWLEDGE==1)
			{
				print "OK"
			}
		}' $TMP_SERIAL_ANSWER`

		if [ "$SERIAL_ANSWER" = "OK" ]; then
			busybox printf "at*quit\r">$TMP_FIFOIN;
			return 0;
		fi

		if [ "$SERIAL_ANSWER" = "ERROR" ];then
			echo "PM Sanity Error: serial_client:\"$1\" command will not passed"
			busybox printf "at*quit\r">$TMP_FIFOIN;
			return 1;
		fi
		busybox usleep $SERIAL_CLIENT_WAIT_STEP
		WAITING_TIME=`echo | busybox awk -v cur="$WAITING_TIME" -v step="$SERIAL_CLIENT_WAIT_STEP" '{print cur+step;}'`
		if [ $3 -lt $WAITING_TIME ];then
			busybox printf "at*quit\r">$TMP_FIFOIN;
			echo "PM Sanity Error: serial_client no answer command - $1"
			return 1;
		fi
	done
}

##################################################################################
#
# Input:		None
#
# Return value:	None
#
# Description:	Print script usage to script
#
##################################################################################
usages () {
	echo "--time          set the measurement time"
	echo "--tel_number    set telephone number"
	echo "--targets_file  set value from file as target"
}

##################################################################################
#
# Input:		$1 - to check if numeric
#
# Return value:	0 if numeric, 1 otherwise
#
# Description:	verify input is numeric
#
##################################################################################
verify_numeric () {
	answer=`echo | busybox awk -v val="$1" '
	{
		res=val+0;
		if(res==val)
			print "true";
		else
			print "false";
	}'`

	if [ "$answer" = "true" ]; then
		return 0;
	else
		return 1;
	fi
}

##################################################################################
#
# Input:		$1 input parameters for scenarios selecting
#				(example: DRX, GSM, DRX_GSM, WB, WB_DRX, DVFM)
#
# Return value:	None
#
# Description:	select scenarios which will executed
#
##################################################################################
select_scenarios () {

	DRX_SCENARIO=0
	GSM_SCENARIO=0
	DRX_AFTER_GSM_SCENARIO=0
	WB_SCENARIO=0
	DRX_AFTER_WB_SCENARIO=0
	DVFM_SCENARIO=0

	case $1 in
	"DRX")
		DRX_SCENARIO=1
		;;
	"GSM")
		GSM_SCENARIO=1
		;;
	"DRX_GSM")
		GSM_SCENARIO=1
		DRX_AFTER_GSM_SCENARIO=1
		;;
	"WB")
		WB_SCENARIO=1
		;;
	"WB_DRX")
		WB_SCENARIO=1
		DRX_AFTER_WB_SCENARIO=1
		;;
	"DVFM")
		DVFM_SCENARIO=1
		;;
	*)
		echo "PM Sanity Error: Incorrect scenario name"
		return 1
		;;
	esac

	return 0
}

##################################################################################
#
# Input:		$1 input parameters given by the user
#				(example: --tel_number 054 --targets_file targets.txt)
#
# Return value:	None
#
# Description:	Set the following if given:
#				time for each scenario measurement, tel number,
#				target file (previous results), results file (current results)
#				--help will print script usage
#
##################################################################################
check_input_params() {
	while [ 1 ] ; do
	if [ "$1" = "--time" ] ; then
			shift;
			verify_numeric $1
			if [ $? -eq 0 ]; then
				MEASUREMENT_TIME=$1
			else
				echo "PM Sanity Warning: incorrect time value, set default value"
			fi
	elif [ "$1" = "--tel_number" ] ; then
		shift ; TEL_NUMBER="$1"
	elif [ "$1" = "--targets_file" ] ; then
		shift ; TARGETS_FILE="$1"
		set_targets $TARGETS_FILE
	elif [ "$1" = "--help" ] ; then
		usages;
		exit 1
	elif [ "$1" = "--scenario" ] ; then
		shift;
		select_scenarios "$1"
		if [ $? -ne 0 ]; then
			exit 1
		fi
	elif [ -z "$1" ] ; then
		break
	else
		echo "PM Sanity Error: unexpected key" 1>&2
		usages;
		exit 1
	fi
	shift
	done

	echo "MEASUREMENT_TIME = $MEASUREMENT_TIME"
	echo "TEL_NUMBER = $TEL_NUMBER"
}

##################################################################################
#
# Input:		None
#
# Return value:	Total idle time
#
# Description:	Open temp apps dc file and parse to get total idle value
#
##################################################################################
get_apps_total_idle() {
	busybox awk '
	{
		i=match($0,"idle:");
		if(i)
			s+=substr($0,i+5,5)
	}
	END
	{
		print s
	}' $TMP_APPS_DUTYCYCLE
}

##################################################################################
#
# Input:		None
#
# Return value:	D2 idle time
#
# Description:	Open temp apps dc file and parse to get D2 value
#
##################################################################################
get_apps_d2_idle() {
	busybox awk '
	{
		i=match($0,"D2");
		if(i)
		{
			w=match($0,"idle:");
			if(w)
				s=substr($0,w+5,5)
		}
	}
	END {print s}' $TMP_APPS_DUTYCYCLE
}

##################################################################################
#
# Input:		None
#
# Return value:	CG idle time
#
# Description:	Open temp apps dc file and parse to get CG value
#
##################################################################################
get_apps_CG_idle() {
	busybox awk '
	{
		i=match($0,"CG");
		if(i)
		{
			w=match($0,"idle:");
			if(w)
				str=substr($0,w+5,5)
		}
	}
	END {print str}' $TMP_APPS_DUTYCYCLE
}

##################################################################################
#
# Input:		None
#
# Return value:	Total timers value
#
# Description:	Open temp apps timers file and parse to get total timers value
#
##################################################################################
get_timer_stats() {
	busybox awk '
	{
		if(match($0, "total events,"))
		{
			i=$4*100;
			i=int(i)/100;
			print i;
		}
	}' $TMP_TIMERS_STATS
}

##################################################################################
#
# Input:		None
#
# Return value:	None
#
# Description:	Set idle pp and value vars
#
##################################################################################
set_dsp_pp_and_value() {
	DSP_IDLE_PP=$1
	DSP_IDLE_VALUE=$2
}

##################################################################################
#
# Input:		None
#
# Return value:	None
#
# Description:	Open temp comm dc file and parse to get dsp pp and idle time
#				into vars
#
##################################################################################
get_pp_dsp_idle() {
	PP_DSP_IDLE=`busybox awk '
	BEGIN
	{
		i=1;
		quantity=0;
		pp_counter=0;
		res=0;
	}
	{
		if(i == 1)
		{
			if(match($0, "Duty Cycle COMM:"))
				i=2;
		}
		else if (i == 2)
		{
			if(match($0, "Duty Cycle DSP:"))
				i=3;
		}
		else if (i == 3)
		{
			if(match($0, "PP"))
			{
				if(match($0,"idle:"))
				{
					s_pp[pp_counter]=match($0, "PP");
					s_pp[pp_counter]=substr($0,s_pp[pp_counter],3);
					res[pp_counter]=$3;
					pp_counter+=1;
					sub(/%$/, "");
					if($3 > 0)
					{
						answer=pp_counter-1;
						quantity+=1;
					}
				}
			}
		}
	}
	END
	{
		if(quantity == 1)
		{
			print s_pp[answer];
			sub(/%$/, "", res[answer]);
			print res[answer];
		}
		else
		{
			for (i = 0; i < pp_counter-1; i++)
			{
				printf s_pp[i] ","
			}
			print s_pp[i]
			for (i = 0; i < pp_counter-1; i++)
			{
				printf res[i]  "\%,"
			}
			print res[i]
		}
	}' $TMP_COMM_DUTYCYCLE`
	set_dsp_pp_and_value $PP_DSP_IDLE
}

##################################################################################
#
# Input:		$1 - SG idle PP
#				$2 - SG idle time
#
# Return value:	None
#
# Description:	Set idle pp and value vars
#
##################################################################################
set_sg_pp_and_value () {
	SG_IDLE_PP=$1
	SG_IDLE_VALUE=$2
}

##################################################################################
#
# Input:		None
#
# Return value:	None
#
# Description:	Open temp comm dc file and parse to get seagull idle pp and value
#				into vars
#
##################################################################################
get_comm_pp_idle() {
	PP_SG_IDLE=`busybox awk '
	BEGIN
	{
		i=1;
		quantity=0;
		pp_counter=0;
		res=0;
	}
	{
		if(i == 1)
		{
			if(match($0, "Duty Cycle COMM:"))
				i=2;
		}
		else if (i == 2)
		{
			if(match($0, "PP"))
			{
				if(match($0,"idle:"))
				{
					s_pp[pp_counter]=match($0, "PP");
					s_pp[pp_counter]=substr($0,s_pp[pp_counter],3);
					res[pp_counter]=$3;
					pp_counter+=1;
					sub(/%$/, "");
					if($3 > 0)
					{
						answer=pp_counter-1;
						quantity+=1;
					}
				}
			}
			if(match($0, "Duty Cycle DSP:"))
				i=3;
		}
	}
	END
	{
		if(quantity == 1)
		{
			print s_pp[answer];
			sub(/%$/, "", res[answer]);
			print res[answer];
		}
		else
		{
			for (i = 0; i < pp_counter-1; i++)
			{
				printf s_pp[i] ","
			}
			print s_pp[i]
			for (i = 0; i < pp_counter-1; i++)
			{
				printf res[i]  "\%,"
			}
			print res[i]
		}
	}' $TMP_COMM_DUTYCYCLE`
	set_sg_pp_and_value $PP_SG_IDLE
}

##################################################################################
#
# Input:		None
#
# Return value:	D2 idle time
#
# Description:	Open temp comm dc file and parse to get D2 value
#
##################################################################################
get_comm_d2_idle() {
	busybox awk '
	BEGIN
	{
		i=1;
	}
	{
		if(i == 1)
		{
			if(match($0, "Duty Cycle COMM:"))
				i=2;
		}
		else if (i == 2)
		{
			if(match($0, "D2"))
			{
				sub(/%$/, "");
				if(match($0,"idle:"))
					print $3;
			}
			if(match($0, "Duty Cycle DSP:"))
				i=3;
		}
	}' $TMP_COMM_DUTYCYCLE
}

##################################################################################
#
# Input:		None
#
# Return value:	None
#
# Description:	Sleep and get D2 idle time during this time
#
##################################################################################
check_apps_D2 () {
	echo "Start measurement"
	echo 1 > /sys/devices/system/cpu/cpu0/stats

	sleep $MEASUREMENT_TIME

	echo "Stop and get results"
	cat /sys/devices/system/cpu/cpu0/duty_cycle > $TMP_APPS_DUTYCYCLE
	cat $TMP_APPS_DUTYCYCLE >>$FULL_LOG_FILE
	cat /sys/devices/system/cpu/cpu0/trace >>$FULL_LOG_FILE

	# parse results
	AFTER_VC_APPS_D2_IDLE_RESAULT=`get_apps_d2_idle`
}

##################################################################################
#
# Input:		None
#
# Return value:	Active OP list
#
# Description:	Create OP list based on MAX PP
#
##################################################################################
get_active_op_list () {
	busybox awk '
	BEGIN
	{
		i=0;
		str_number=0;
	}
	{
		if(i==0)
		{
			if(match($0, "name:"))
			{
				if((match($0,"D1")) || (match($0,"D2")) || (match($0,"CG")))
				{
					i=1;
				} else
				{
					print str_number;
					str_number+=1;
				}
			}
		}
	}' $TMP_OPS
}

##################################################################################
#
# Input:		None
#
# Return value:	OP list
#
# Description:	Finds all enabled ops and add them to list
#
##################################################################################
get_available_op () {
	busybox awk '
	BEGIN
	{
		i=0;
		str_number=0;
	}
	{
		if(i==0)
		{
			if(match($0, "name:"))
			{
				if((match($0,"D1")) || (match($0,"D2")) || (match($0,"CG")))
				{
					i=1;
				} else
				{
					if(match($0,"Enabled"))
						print str_number
					str_number+=1;
				}
			}
		}
	}' $TMP_OPS
}

##################################################################################
#
# Input:		None
#
# Return value:	None
#
# Description:	Enable all ops by user
#
##################################################################################
enable_all_ops () {
	for OP in $ENABLED_OPS_LIST
	do
		echo $OP,1 > /sys/devices/system/cpu/cpu0/enable_op
	done
}

##################################################################################
#
# Input:		None
#
# Return value:	None
#
# Description:	Set DVFM result for each pp in a temp result file
#
##################################################################################
dvfm_blocked_pp_to_result_output () {
	ACTIVE_OP_LIST=`get_active_op_list`

	for OP in $ACTIVE_OP_LIST
	do
		RESULT=`busybox awk  -v op=$OP '
		{
			if(match($0, op))
			{
				print $0
			}
		}' $TMP_INTERM_DVFM_TEST_ANSWER`
		if [ "$RESULT" = "" ]; then
			echo "DVFM APPS PP $OP           UNKNOWN" >>$TMP_DVFM_TEST_ANSWER
		else
			echo "$RESULT" >>$TMP_DVFM_TEST_ANSWER
		fi
		done
}

##################################################################################
#
# Input:		None
#
# Return value:	0 on success, 1 otherwise
#
# Description:	DVFM scenario
#				For each not dissabled PP disable rest of active PPs and verify that
#				system has entered this PP
#
##################################################################################
DVFM_scenario (){
	cat /sys/devices/system/cpu/cpu0/ops > $TMP_OPS
	ENABLED_OPS_LIST=`get_available_op`

	RESULT="PASS"
	OP_LIST2=$ENABLED_OPS_LIST
	for OP1 in $ENABLED_OPS_LIST
	do
		echo "Test OP $OP1"
		echo "Test OP $OP1: " >>$FULL_LOG_FILE

		echo $OP1,1 > /sys/devices/system/cpu/cpu0/enable_op
		for OP2 in $OP_LIST2
		do
			if [ "$OP1" -ne "$OP2" ]; then
				echo $OP2,0 > /sys/devices/system/cpu/cpu0/enable_op
			fi
		done

		cat /sys/devices/system/cpu/cpu0/op >$TMP_OP
		cat $TMP_OP >>$FULL_LOG_FILE
		echo "" >>$FULL_LOG_FILE
		cat /sys/devices/system/cpu/cpu0/trace >>$FULL_LOG_FILE

		# find pointed OP and form answer string
		OP_ACTIVITY_ANSWER=`busybox awk -v op=$OP1 '
		{
			if(match($0, "OP:"))
			{
				s=sprintf ("OP:%s", op);
				if(match($0, s))
				{
					print "DVFM APPS PP", op, "          PASS";
				}
				else
				{
					print "DVFM APPS PP", op, "          CHECK";
				}
			}
		}' $TMP_OP`
		echo "$OP_ACTIVITY_ANSWER" >>$TMP_INTERM_DVFM_TEST_ANSWER

		# set result var for all pp test indication
		if [ "$RESULT" = "PASS" ]; then
			RESULT=`echo "$OP_ACTIVITY_ANSWER" | busybox awk '
			{
				if(match($0, "FAIL"))
				{
					print "FAIL"
				}
				else
				{
					print "PASS"
				}
			}'`
		fi
	done

	enable_all_ops
	dvfm_blocked_pp_to_result_output

	return 0;
}

##################################################################################
#
# Input:		None
#
# Return value:	None
#
# Description:	Start following debug tools:
#				1. Apps duty cycle (started by default, only reset)
#				2. Comm duty cycle
#				3. Apps timers
##################################################################################
start_measurements()
{
	# comm dc
	$COMMPM_ACC_PATH/commpm_acc dc_start >/dev/null
	$COMMPM_ACC_PATH/commpm_acc dc_reset >/dev/null

	# apps dc
	echo 1 > /sys/devices/system/cpu/cpu0/stats

	# timers
	echo 0 > /proc/timer_stats
	echo 1 > /proc/timer_stats
}

##################################################################################
#
# Input:		None
#
# Return value:	None
#
# Description:	Stop started debug tools, and save outputs to temp files and log
#
##################################################################################
stop_and_get_measurements()
{
	# comm dc
	$COMMPM_ACC_PATH/commpm_acc dc_stop >/dev/null
	$COMMPM_ACC_PATH/commpm_acc dc_display >$TMP_COMM_DUTYCYCLE
	cat  $TMP_COMM_DUTYCYCLE >>$FULL_LOG_FILE

	# apps dc
	cat /sys/devices/system/cpu/cpu0/duty_cycle > $TMP_APPS_DUTYCYCLE
	cat  $TMP_APPS_DUTYCYCLE >>$FULL_LOG_FILE
	cat /sys/devices/system/cpu/cpu0/trace >>$FULL_LOG_FILE

	# timers
	cat /proc/timer_stats > $TMP_TIMERS_STATS
	cat  $TMP_TIMERS_STATS >>$FULL_LOG_FILE
}

##################################################################################
#
# Input:		None
#
# Return value:	0 on success, 1 otherwise
#
# Description:	DRX scenario
#				Sleep for measured time while collecting debug info
#				Save results in vars to be compared and displayed later
#
##################################################################################
DRX_scenario() {

	echo "Start measurement"
	start_measurements

	sleep $MEASUREMENT_TIME

	echo "Stop and get results"
	stop_and_get_measurements

	# parse data
	DRX_APPS_D2_IDLE_RESAULT=`get_apps_d2_idle`
	DRX_APPS_TOTAL_IDLE_RESAULT=`get_apps_total_idle`
	DRX_APPS_TIMERS_RESULT=`get_timer_stats`
	DRX_COMM_D2_IDLE_RESULT=`get_comm_d2_idle`

	get_comm_pp_idle
	DRX_SG_IDLE_PP_RESAULT=$SG_IDLE_PP
	DRX_SG_IDLE_VALUE_RESAULT=$SG_IDLE_VALUE

	get_pp_dsp_idle
	DRX_DSP_IDLE_PP_RESAULT=$DSP_IDLE_PP
	DRX_DSP_IDLE_VALUE_RESAULT=$DSP_IDLE_VALUE

	return 0
}

##################################################################################
#
# Input:		$1 - value 1
#				$2 - value 2
#				$5 - flag to indicate fatal scenarios
#
# Return value:	Comparison result for $1 < $2
#
# Description:	Checks input values (must be numeric)
#				Compare the two numeric values
#
##################################################################################
check_result_timer() {
	ANSWER=`echo | busybox awk -v value1="$1" -v value2="$2" -v value3="$3" '
			{
				tmp1=value1+0;
				tmp2=value2+0;
				tmp3=value3+0;
				if((tmp1 == value1) && (tmp2 == value2) && (tmp3 == value3))
					print "PASS";
				else
					print "FAIL";
			}'`

	if [ "$ANSWER" = "PASS" ]; then
		RESULT=`echo | busybox awk -v value="$1" -v target="$2" -v fail_target="$3" '
		{
			if(value <= target)
				print "PASS";
			else if(value >= fail_target)
				print "FAIL";
			else
				print "CHECK";
		}'`
	else
		RESULT="FAIL"
	fi
	busybox printf "%-23s %5s  %32s%s  %8s%s\n" "$4" $RESULT "$1" "$5" "$2" "$5" >>$FINAL_RESUALTS_FILE

	if [ "$6" = "FATAL" -a "$RESULT" = "FAIL" ]; then
		SANITY_RESULT="FAIL"
	fi
}

##################################################################################
#
# Input:		$1 - value 1
#				$2 - value 2
#				$5 - flag to indicate fatal scenarios
#
# Return value:	Comparison result for $1 > $2
#
# Description:	Checks input values (must be numeric)
#				Compare the two numeric values
#
##################################################################################
check_result() {
	ANSWER=`echo | busybox awk -v value1="$1" -v value2="$2" -v value3="$3" '
			{
				tmp1=value1+0;
				tmp2=value2+0;
				tmp3=value3+0;
				if((tmp1 == value1) && (tmp2 == value2) && (tmp3 == value3))
					print "PASS";
				else
					print "FAIL";
			}'`

	if [ "$ANSWER" = "PASS" ]; then
		RESULT=`echo | busybox awk -v value="$1" -v target="$2" -v fail_target="$3" '
		{
			if(value >= target)
				print "PASS";
			else if(value <= fail_target)
				print "FAIL";
			else
				print "CHECK";
		}'`
	else
		RESULT="FAIL"
	fi

	busybox printf "%-23s %5s  %32s%s  %8s%s\n" "$4" $RESULT "$1" "$5" "$2" "$5" >>$FINAL_RESUALTS_FILE

	if [ "$6" = "FATAL" -a "$RESULT" = "FAIL" ]; then
		SANITY_RESULT="FAIL"
	fi
}

##################################################################################
#
# Input:		$1 - value 1
#				$2 - value 2
#				$4 - flag to indicate fatal scenarios
#
# Return value:	Comparison result for $1 == $2
#
# Description:	Compare the two values
#
##################################################################################
check_result_pp() {
	if [ "$1" = "$2" ]; then
		RESULT="PASS"
	else
		RESULT="FAIL"
	fi
	busybox printf "%-23s %5s  %32s   %8s\n" "$3" $RESULT "$1" "$2" >>$FINAL_RESUALTS_FILE

	if [ "$4" = "FATAL" -a "$RESULT" = "FAIL" ]; then
		SANITY_RESULT="FAIL"
	fi
}

##################################################################################
#
# Input:		None
#
# Return value:	None
#
# Description:	Prepare final result file
#
##################################################################################
prepare_result_file () {
	echo "---------------------------------------------------------------------------" >>$FINAL_RESUALTS_FILE
	echo "TEST_NAME              PASS/FAIL                          RESULT     TARGET" >$FINAL_RESUALTS_FILE
	echo "---------------------------------------------------------------------------" >>$FINAL_RESUALTS_FILE

	# DRX
	if [ $DRX_SCENARIO -eq 1 ] ; then
		check_result $DRX_APPS_D2_IDLE_RESAULT $DRX_TARGET_APPS_D2_IDLE $DRX_FAIL_TARGET_APPS_D2_IDLE "$DRX_NAME_APPS_D2_IDLE" "%" "FATAL"
		check_result $DRX_APPS_TOTAL_IDLE_RESAULT $DRX_TARGET_APPS_TOTAL_IDLE $DRX_FAIL_TARGET_APPS_TOTAL_IDLE "$DRX_NAME_APPS_TOTAL_IDLE" "%" "NORMAL"
		check_result $DRX_COMM_D2_IDLE_RESULT $DRX_TARGET_COMM_D2_IDLE $DRX_FAIL_TARGET_COMM_D2_IDLE "$DRX_NAME_COMM_SG_D2_IDLE" "%" "NORMAL"
		check_result_timer $DRX_APPS_TIMERS_RESULT $DRX_TARGET_APPS_TIMERS $DRX_FAIL_TARGET_APPS_TIMERS "$DRX_NAME_APPS_TIMERS" " " "NORMAL"
		check_result_pp $DRX_SG_IDLE_PP_RESAULT $DRX_TARGET_COMM_SG_PP "$DRX_NAME_COMM_SG_PP" "NORMAL"
		check_result $DRX_SG_IDLE_VALUE_RESAULT $DRX_TARGET_COMM_SG_PP_IDLE $DRX_FAIL_TARGET_COMM_SG_PP_IDLE "$DRX_NAME_COMM_SG_PP_IDLE" "%" "NORMAL"
		check_result_pp $DRX_DSP_IDLE_PP_RESAULT $DRX_TARGET_COMM_DSP_PP "$DRX_NAME_COMM_DSP_PP" "NORMAL"
		check_result $DRX_DSP_IDLE_VALUE_RESAULT $DRX_TARGET_COMM_DSP_PP_IDLE $DRX_FAIL_TARGET_COMM_DSP_PP_IDLE "$DRX_NAME_COMM_DSP_PP_IDLE" "%" "NORMAL"
	echo "" >>$FINAL_RESUALTS_FILE
	fi

	# GSM
	if [ $GSM_SCENARIO -eq 1 ] ; then
		if [ $GSM_SCENARIO_FAIL -eq 0 ];then
			check_result $GSM_APPS_CG_IDLE_RESAULT $GSM_TARGET_APPS_CG_IDLE $GSM_FAIL_TARGET_APPS_CG_IDLE "$GSM_NAME_APPS_CG_IDLE" "%" "FATAL"
			check_result $GSM_APPS_TOTAL_IDLE_RESAULT $GSM_TARGET_APPS_TOTAL_IDLE $GSM_FAIL_TARGET_APPS_TOTAL_IDLE "$GSM_NAME_APPS_TOTAL_IDLE" "%" "NORMAL"
			check_result_timer $GSM_APPS_TIMERS_RESAULT $GSM_TARGET_APPS_TIMERS $GSM_FAIL_TARGET_APPS_TIMERS "$GSM_NAME_APPS_TIMERS" " " "NORMAL"
			check_result_pp $GSM_SG_IDLE_PP_RESAULT $GSM_TARGET_COMM_SG_PP "$GSM_NAME_COMM_SG_PP" "NORMAL"
			check_result $GSM_SG_IDLE_VALUE_RESAULT $GSM_TARGET_COMM_SG_PP_IDLE $GSM_FAIL_TARGET_COMM_SG_PP_IDLE "$GSM_NAME_COMM_SG_PP_IDLE" "%" "NORMAL"
			check_result_pp $GSM_DSP_IDLE_PP_RESAULT $GSM_TARGET_COMM_DSP_PP "$GSM_NAME_COMM_DSP_PP" "NORMAL"
			check_result $GSM_DSP_IDLE_VALUE_RESAULT $GSM_TARGET_COMM_DSP_PP_IDLE $GSM_FAIL_TARGET_COMM_DSP_PP_IDLE "$GSM_NAME_COMM_DSP_PP_IDLE" "%" "NORMAL"
		else
			echo "GSM SCENARIO FAILED" >>$FINAL_RESUALTS_FILE
		fi
		echo "" >>$FINAL_RESUALTS_FILE
	fi

	# DRX after GSM
	if [ $DRX_AFTER_GSM_SCENARIO -eq 1 ] ; then
		check_result $DRX_AFTER_GSM_RESAULT $DRX_TARGET_APPS_D2_IDLE $DRX_FAIL_TARGET_APPS_D2_IDLE "$DRX_AFTER_GSM_NAME" "%" "FATAL"
		echo "" >>$FINAL_RESUALTS_FILE
	fi

	# WB
	if [ $WB_SCENARIO -eq 1 ] ; then
		if [ $WB_SCENARIO_FAIL -eq 0 ];then
			check_result $WB_APPS_CG_IDLE_RESAULT $WB_TARGET_APPS_CG_IDLE $WB_FAIL_TARGET_APPS_CG_IDLE "$WB_NAME_APPS_CG_IDLE" "%" "FATAL"
			check_result $WB_APPS_TOTAL_IDLE_RESAULT $WB_TARGET_APPS_TOTAL_IDLE $WB_FAIL_TARGET_APPS_TOTAL_IDLE "$WB_NAME_APPS_TOTAL_IDLE" "%" "NORMAL"
			check_result_timer $WB_APPS_TIMERS_RESAULT $WB_TARGET_APPS_TIMERS $WB_FAIL_TARGET_APPS_TIMERS "$WB_NAME_APPS_TIMERS" " " "NORMAL"
			check_result_pp $WB_SG_IDLE_PP_REAULT $WB_TARGET_COMM_SG_PP "$WB_NAME_COMM_SG_PP" "NORMAL"
			check_result $WB_SG_IDLE_VALUE_RESAULT $WB_TARGET_COMM_SG_PP_IDLE $WB_FAIL_TARGET_COMM_SG_PP_IDLE "$WB_NAME_COMM_SG_PP_IDLE" "%" "NORMAL"
			check_result_pp $WB_DSP_IDLE_PP_RESAULT $WB_TARGET_COMM_DSP_PP "$WB_NAME_COMM_DSP_PP" "NORMAL"
			check_result $WB_DSP_IDLE_VALUE_RESAULT $WB_TARGET_COMM_DSP_PP_IDLE $WB_FAIL_TARGET_COMM_DSP_PP_IDLE "$WB_NAME_COMM_DSP_PP_IDLE" "%" "NORMAL"
		else
			echo "WB SCENARIO FAILED" >>$FINAL_RESUALTS_FILE
		fi
		echo "" >>$FINAL_RESUALTS_FILE
	fi

	#DRX after WB
	if [ $DRX_AFTER_WB_SCENARIO -eq 1 ] ; then
		check_result $DRX_AFTER_WB_RESAULT $DRX_TARGET_APPS_D2_IDLE $DRX_FAIL_TARGET_APPS_D2_IDLE "$DRX_AFTER_WB_NAME" "%" "FATAL"
		echo "" >>$FINAL_RESUALTS_FILE
	fi

	#DVFM
	if [ $DVFM_SCENARIO -eq 1 ] ; then
		cat $TMP_DVFM_TEST_ANSWER>>$FINAL_RESUALTS_FILE
		echo "" >>$FINAL_RESUALTS_FILE
	fi

	echo "PM Sanity test took $TEST_DURATION seconds" >>$FINAL_RESUALTS_FILE
	echo "---------------------------------------------------------------------------" >>$FINAL_RESUALTS_FILE

	echo "############################################################################" >>$FINAL_RESUALTS_FILE
	echo "##                       PM SANITY RESULT: $SANITY_RESULT                           ##" >>$FINAL_RESUALTS_FILE
	echo "############################################################################" >>$FINAL_RESUALTS_FILE
}

##################################################################################
#
# Input:		None
#
# Return value:	None
#
# Description:	Remove temporary files
#
##################################################################################
remove_tmp_files () {
	if [ -f $TMP_FIFOIN ]; then
		rm $TMP_FIFOIN
	fi
	if [ -f $TMP_APPS_DUTYCYCLE ]; then
		rm $TMP_APPS_DUTYCYCLE
	fi
	if [ -f $TMP_TIMERS_STATS ]; then
		rm $TMP_TIMERS_STATS
	fi
	if [ -f $TMP_COMM_DUTYCYCLE ]; then
		rm $TMP_COMM_DUTYCYCLE
	fi
	if [ -f $TMP_SERIAL_ANSWER ]; then
		rm $TMP_SERIAL_ANSWER
	fi
	if [ -f $TMP_OP ]; then
		rm $TMP_OP
	fi
	if [ -f $TMP_DVFM_TEST_ANSWER ]; then
		rm $TMP_DVFM_TEST_ANSWER
	fi
	if [ -f $TMP_OPS ]; then
		rm $TMP_OPS
	fi
	if [ -f $TMP_INTERM_DVFM_TEST_ANSWER ]; then
		rm $TMP_INTERM_DVFM_TEST_ANSWER
	fi
}

##################################################################################
#
# Input:		None
#
# Return value:	0 on success, 1 otherwise
#
# Description:	GSM scenario
#				Make a GSM VC while collecting debug info
#				Save results in vars to be compared and displayed later
#
##################################################################################
GSM_scenario () {
	echo "Make VC"

	# at*band=0
	serial_command "at*band=0" "OK" 20000000 "ERROR"
	if [ $? -ne 0 ]; then
		echo "PM Sanity Error: at*band=0 (GSM)"
		return 1
	fi

	# make voice call
	serial_command "atd$TEL_NUMBER;" "CONNECT" $CALL_WAIT_TIME "NO CARRIER" "ERROR"
	if [ $? -ne 0 ]; then
		echo "PM Sanity Error: atd"
		serial_command "ath" "NO CARRIER" $ATH_WAIT_TIME "ERROR"
		return 1
	fi

	echo "Start measurement"
	start_measurements

	sleep $MEASUREMENT_TIME

	echo "Stop and get results"
	stop_and_get_measurements

	# parse data
	GSM_APPS_CG_IDLE_RESAULT=`get_apps_CG_idle`
	GSM_APPS_TOTAL_IDLE_RESAULT=`get_apps_total_idle`
	GSM_APPS_TIMERS_RESAULT=`get_timer_stats`

	get_comm_pp_idle
	GSM_SG_IDLE_PP_RESAULT=$SG_IDLE_PP
	GSM_SG_IDLE_VALUE_RESAULT=$SG_IDLE_VALUE

	get_pp_dsp_idle
	GSM_DSP_IDLE_PP_RESAULT=$DSP_IDLE_PP
	GSM_DSP_IDLE_VALUE_RESAULT=$DSP_IDLE_VALUE

	echo "Hang VC"

	# hang vc
	serial_command "ath" "NO CARRIER" $ATH_WAIT_TIME "ERROR"
	if [ $? -ne 0 ]; then
		echo "Error: ath"
		return 1
	fi

	return 0;
}

##################################################################################
#
# Input:		None
#
# Return value:	0 on success, 1 otherwise
#
# Description:	WB scenario
#				Make a WB VC while collecting debug info
#				Save results in vars to be compared and displayed later
#
##################################################################################
WB_scenario () {
	echo "Make VC"

	# at*band=1
	serial_command "at*band=1" "OK" 20000000 "ERROR"
	if [ $? -ne 0 ]; then
		echo "PM Sanity Error: at*band=1 (WB)"
		return 1
	fi

	# make voice call
	serial_command "atd$TEL_NUMBER;" "CONNECT" $CALL_WAIT_TIME "NO CARRIER" "ERROR"
	if [ $? -ne 0 ]; then
		echo "PM Sanity Error: atd"
		serial_command "ath" "NO CARRIER" $ATH_WAIT_TIME "ERROR"
		return 1
	fi

	echo "Start measurement"
	start_measurements

	# sleep
	sleep $MEASUREMENT_TIME

	echo "Stop and get results"
	stop_and_get_measurements

	# parse data
	WB_APPS_CG_IDLE_RESAULT=`get_apps_CG_idle`
	WB_APPS_TOTAL_IDLE_RESAULT=`get_apps_total_idle`
	WB_APPS_TIMERS_RESAULT=`get_timer_stats`

	get_comm_pp_idle
	WB_SG_IDLE_PP_REAULT=$SG_IDLE_PP
	WB_SG_IDLE_VALUE_RESAULT=$SG_IDLE_VALUE

	get_pp_dsp_idle
	WB_DSP_IDLE_PP_RESAULT=$DSP_IDLE_PP
	WB_DSP_IDLE_VALUE_RESAULT=$DSP_IDLE_VALUE

	echo "Hang VC"

	# hang vc
	serial_command "ath" "NO CARRIER" $ATH_WAIT_TIME "ERROR"
	if [ $? -ne 0 ]; then
		echo "PM Sanity Error: ath"
		return 1
	fi

	return 0;
}

###################################################################################
############################ Start of script ######################################
###################################################################################

echo ""
echo "Start of script"
echo ""

START_TIME=$(date +%s)

# wait for uart constraints to be removed
sleep 5

# check input parameters and set vars
check_input_params $@

busybox mkfifo $TMP_FIFOIN

############################# REGISTER TO NETWORK #################################
echo "Register to network"

# at
serial_command "at" "OK" 1000000 "ERROR"
if [ $? -ne 0 ]; then
	echo "PM Sanity Error: at"
	remove_tmp_files
	exit 1
fi

# at+creg?
serial_command "at+creg?" "CREG:" 1000000 "ERROR"
if [ $? -ne 0 ]; then
	echo "PM Sanity Error: at+creg?"
	remove_tmp_files
	exit 1
fi

# at+cpin?
serial_command "at+cpin?" "READY" 1000000 "ERROR"
if [ $? -ne 0 ]; then
	echo "PM Error: at+cpin?"
	remove_tmp_files
	exit 1
fi
################################# DRX #############################################
if [ $DRX_SCENARIO -eq 1 ] ; then
	echo ""
	echo "DRX scenario"
	echo "-------------------"

	echo "" >$FULL_LOG_FILE
	echo "DRX scenario" >>$FULL_LOG_FILE
	echo "-------------------" >>$FULL_LOG_FILE

	DRX_scenario

	if [ $? -eq 1 ]; then
		echo ""
		echo "PM Sanity Error: DRX scenario failed"
	fi
fi

############################ GSM VC ###############################################
if [ $GSM_SCENARIO -eq 1 ] ; then
	echo ""
	echo "GSM scenario"
	echo "-------------------"

	echo "" >>$FULL_LOG_FILE
	echo "GSM scenario" >>$FULL_LOG_FILE
	echo "-------------------" >>$FULL_LOG_FILE

	GSM_scenario
	GSM_SCENARIO_FAIL=$?

	if [ $GSM_SCENARIO_FAIL -eq 1 ]; then
		echo ""
		echo "PM Sanity Error: GSM scenario failed"
	else
		sleep 10
	fi
fi

############################ BACK TO DRX ##########################################
if [ $DRX_AFTER_GSM_SCENARIO -eq 1 ] ; then
	echo ""
	echo "Return to DRX scenario"
	echo "-------------------"

	echo "" >>$FULL_LOG_FILE
	echo "Return to DRX scenario" >>$FULL_LOG_FILE
	echo "-------------------" >>$FULL_LOG_FILE

	check_apps_D2
	DRX_AFTER_GSM_RESAULT=$AFTER_VC_APPS_D2_IDLE_RESAULT
fi

############################ WB VC ################################################
if [ $WB_SCENARIO -eq 1 ] ; then
	echo ""
	echo "WB scenario"
	echo "-------------------"

	echo "" >>$FULL_LOG_FILE
	echo "WB scenario" >>$FULL_LOG_FILE
	echo "-------------------" >>$FULL_LOG_FILE

	WB_scenario
	WB_SCENARIO_FAIL=$?

	if [ $WB_SCENARIO_FAIL -eq 1 ]; then
		echo ""
		echo "PM Sanity Error: WB scenario failed"
	else
		sleep 10
	fi
fi

############################ BACK TO DRX ##########################################
if [ $DRX_AFTER_WB_SCENARIO -eq 1 ] ; then
	echo ""
	echo "Return to DRX scenario"
	echo "-------------------"

	echo "" >>$FULL_LOG_FILE
	echo "Return to DRX scenario" >>$FULL_LOG_FILE
	echo "-------------------" >>$FULL_LOG_FILE

	check_apps_D2
	DRX_AFTER_WB_RESAULT=$AFTER_VC_APPS_D2_IDLE_RESAULT
fi

############################ DVFM #################################################
if [ $DVFM_SCENARIO -eq 1 ] ; then
	echo ""
	echo "Apps DVFM scenario"
	echo "-------------------"

	echo "" >>$FULL_LOG_FILE
	echo "Apps DVFM scenario" >>$FULL_LOG_FILE
	echo "-------------------" >>$FULL_LOG_FILE

	DVFM_scenario

	if [ $? -eq 1 ]; then
		echo ""
		echo "PM Sanity Error: DVFM scenario failed"
	fi
fi

##################################################################################

# return to at*band=4 (WB+GSM+LTE)
serial_command "at*band=4" "OK" 20000000 "ERROR"
if [ $? -ne 0 ]; then
	echo "Error: at*band=4 (WB+GSM+LTE)"
fi

# generate result file
echo ""
echo ""

END_TIME=$(date +%s)
TEST_DURATION=$(( $END_TIME - $START_TIME ))

prepare_result_file
cat $FINAL_RESUALTS_FILE

#remove temporary files
remove_tmp_files

echo "End of test"
exit 0
